
import 'dart:async';

import 'package:flutter/foundation.dart';
import 'package:flutter/gestures.dart';
import 'package:flutter/material.dart';
import 'package:flutter/rendering.dart';
import 'package:flutter/services.dart';
import 'package:flutter/widgets.dart';

typedef void WebViewCreatedCallback(BridgeWebViewController controller);

enum JavascriptMode {
    disabled,
    unrestricted,
}

typedef void PageFinishedCallback(String url);
typedef BridgeHandlerCallback = Future<String> Function(String args);

final RegExp _validChannelNames = RegExp('^[a-zA-Z_][a-zA-Z0-9]*\$');

class BridgeWebView extends StatefulWidget {

    const BridgeWebView({
        Key key,
        @required this.url,
        this.hidden = false,
        this.onPageFinished,
        this.bridgeChannels
    }) : super(key: key);

    final String url;
    final bool hidden;
    final PageFinishedCallback onPageFinished;
    final Set<BridgeChannel> bridgeChannels;

    @override
    State<StatefulWidget> createState() => _BridgeWebViewState();
}

class _BridgeWebViewState extends State<BridgeWebView> {
    final webviewController = BridgeWebViewController();
    Rect _rect;
    Timer _resizeTimer;

    var _onDestroy;

    @override
    void initState() {
        super.initState();
    }

    @override
    void dispose() {
        super.dispose();
        _onDestroy?.cancel();
        _resizeTimer?.cancel();
        webviewController.hide();
        // webviewController.dispose();
    }

    @override
    Widget build(BuildContext context) {
        return Scaffold(
            resizeToAvoidBottomInset: false,
            body: _WebviewPlaceholder(
                onRectChanged: (Rect value) {
                    webviewController.launch(
                        widget.url,
                        hidden: widget.hidden,
                        bridgeChannels: widget.bridgeChannels,
                        onPageFinished: widget.onPageFinished,
                    );
                },
                child: const Center(child: const CircularProgressIndicator()),
            ),
        );
    }

}

class _WebviewPlaceholder extends SingleChildRenderObjectWidget {
    const _WebviewPlaceholder({
        Key key,
        @required this.onRectChanged,
        Widget child,
    }) : super(key: key, child: child);

    final ValueChanged<Rect> onRectChanged;

    @override
    RenderObject createRenderObject(BuildContext context) {
        return _WebviewPlaceholderRender(
            onRectChanged: onRectChanged,
        );
    }

    @override
    void updateRenderObject(BuildContext context, _WebviewPlaceholderRender renderObject) {
        renderObject..onRectChanged = onRectChanged;
    }
}

class _WebviewPlaceholderRender extends RenderProxyBox {
    _WebviewPlaceholderRender({
        RenderBox child,
        ValueChanged<Rect> onRectChanged,
    })  : _callback = onRectChanged,
            super(child);

    ValueChanged<Rect> _callback;
    Rect _rect;

    Rect get rect => _rect;

    set onRectChanged(ValueChanged<Rect> callback) {
        if (callback != _callback) {
            _callback = callback;
            notifyRect();
        }
    }

    void notifyRect() {
        if (_callback != null && _rect != null) {
            _callback(_rect);
        }
    }

    @override
    void paint(PaintingContext context, Offset offset) {
        super.paint(context, offset);
        final rect = offset & size;
        if (_rect != rect) {
            _rect = rect;
            notifyRect();
        }
    }
}

Set<String> _extractChannelNames(Set<BridgeChannel> channels) {
    final Set<String> channelNames = channels == null
        ? Set<String>()
        : channels.map((BridgeChannel channel) => channel.name).toSet();
    return channelNames;
}

class BridgeChannel {
    BridgeChannel({
        @required this.name,
        @required this.onBridgeHandler,
    })
        : assert(name != null),
            assert(onBridgeHandler != null),
            assert(_validChannelNames.hasMatch(name));

    final String name;
    final BridgeHandlerCallback onBridgeHandler;
}

class BridgeWebViewController {
    factory BridgeWebViewController() => _instance ??= BridgeWebViewController._();

    static BridgeWebViewController _instance;

    final MethodChannel _channel;

    Map<String, BridgeChannel> _bridgeChannels = <String, BridgeChannel>{};

    final _onDestroy = StreamController<Null>.broadcast();
    Stream<Null> get onDestroy => _onDestroy.stream;

    PageFinishedCallback _onPageFinished;

    BridgeWebViewController._()
        : _channel = MethodChannel('com.microduino.flutter/bridgewebview') {
        _channel.setMethodCallHandler(_onMethodCall);
    }

    void _updateBridgeChannelsFromSet(Set<BridgeChannel> channels) {
        _bridgeChannels.clear();
        if (channels == null) {
            return;
        }
        for (BridgeChannel channel in channels) {
            _bridgeChannels[channel.name] = channel;
        }
    }

    Future<String> _onMethodCall(MethodCall call) async {
        switch (call.method) {
            case 'onDestroy':
                _onDestroy.add(null);
                break;
            case 'onPageFinished':
                final String url = call.arguments['url'];
                if (_onPageFinished != null) {
                    _onPageFinished(url);
                }
                print('----> onPageFinished ${url}');
                return null;
            case 'onCallHandlerCallBack':
                print('----> onCallHandlerCallBack ${call.arguments}');
                return null;
            case 'onRegisterHandlerCallback':
                final String channel = call.arguments['method'];
                final String args = call.arguments['data'];
                print('----> onRegisterHandlerCallback ${channel}, args === ${args}');
                return await _bridgeChannels[channel].onBridgeHandler(args);
        }
        throw MissingPluginException(
            '${call.method} was invoked but has no handler');
    }

    Future<void> launch(String url, {
        Map<String, String> headers,
        bool hidden,
        Set<BridgeChannel> bridgeChannels,
        PageFinishedCallback onPageFinished,
    }) async {
        _updateBridgeChannelsFromSet(bridgeChannels);
        _onPageFinished = onPageFinished;

        final args = <String, dynamic>{
            'url': url,
            'hidden': hidden ?? false,
            'bridgeChannels': _extractChannelNames(bridgeChannels).toList(),
        };

        if (headers != null) {
            args['headers'] = headers;
        }

        await _channel.invokeMethod('launch', args);
    }

    /// resize webview
    Future<void> resize(Rect rect) async {
        final args = {};
        args['rect'] = {
            'left': rect.left,
            'top': rect.top,
            'width': rect.width,
            'height': rect.height,
        };
        await _channel.invokeMethod('resize', args);
    }

    Future<void> close() async => await _channel.invokeMethod('close');

    Future<void> loadUrl(String url) async {
        assert(url != null);
        _validateUrlString(url);
        return _channel.invokeMethod('loadUrl', url);
    }

    Future<void> callHandler(String method, String data) async {
        assert(method != null);
        _channel.invokeMethod(
            'callHandler', {"name": method, "data": data});
    }

    Future<void> registerHandler(BridgeChannel channel) async {
        _bridgeChannels[channel.name] = channel;
        _channel.invokeMethod(
            'registerHandler', {"name": channel.name});
    }

    Future<void> registerBridgeChannels(Set<BridgeChannel> channels) async {
        if (channels == null) {
            return;
        }
        for (BridgeChannel channel in channels) {
            _bridgeChannels[channel.name] = channel;
        }
        _channel.invokeMethod(
            'registerBridgeChannels', _extractChannelNames(channels).toList());
    }

    Future<String> currentUrl() async {
        final String url = await _channel.invokeMethod('currentUrl');
        return url;
    }

    Future<bool> canGoBack() async {
        final bool canGoBack = await _channel.invokeMethod('canGoBack');
        return canGoBack;
    }

    Future<bool> canGoForward() async {
        final bool canGoForward = await _channel.invokeMethod('canGoForward');
        return canGoForward;
    }

    Future<void> goBack() async {
        return _channel.invokeMethod('goBack');
    }

    Future<void> goForward() async {
        return _channel.invokeMethod('goForward');
    }

    Future<void> reload() async {
        return _channel.invokeMethod('reload');
    }

    Future<void> clearCache() async {
        await _channel.invokeMethod('clearCache');
        return reload();
    }

    Future<void> hide() async {
        return _channel.invokeMethod('hide');
    }

    Future<void> show() async {
        return _channel.invokeMethod('show');
    }

    Future<String> readFile(String name) async {
        return await _channel.invokeMethod('readFile', name);
    }

    void dispose() {
        _onDestroy.close();
        _instance = null;
    }

}

void _validateUrlString(String url) {
    try {
        final Uri uri = Uri.parse(url);
        if (uri.scheme.isEmpty) {
            throw ArgumentError('Missing scheme in URL string: "$url"');
        }
    } on FormatException catch (e) {
        throw ArgumentError(e);
    }
}

